#!/bin/bash
# BjAnalyzer Installation Script - Fixed Version (Includes User Application Directory)

# Check if running as root
if [[ $EUID -ne 0 ]]; then
    echo "This installation script requires root privileges. Please run with sudo:"
    echo "sudo $0"
    exit 1
fi

# Define variables
APP_NAME="BjAnalyzer"
EXEC_SOURCE="./BjAnalyzer.bin"
ICON_SOURCE="./icon/ico256.png"  # Updated icon source path
ICON_DIR_SOURCE="./icon"          # Added icon directory variable
INSTALL_DIR="/opt/$APP_NAME"
BIN_DIR="/usr/local/bin"
ICON_DIR="/usr/share/icons/hicolor/256x256/apps"
DESKTOP_DIR="/usr/share/applications"

# Get real username (Important!)
REAL_USER=${SUDO_USER:-$(logname 2>/dev/null)}
if [[ -z "$REAL_USER" ]]; then
    REAL_USER=$(who am i | awk '{print $1}')
fi
if [[ -z "$REAL_USER" ]] || [[ "$REAL_USER" == "root" ]]; then
    echo "Warning: Unable to determine real username, using current user"
    REAL_USER=$(whoami)
fi

REAL_USER_HOME=$(eval echo ~$REAL_USER)

# Check if source files exist
if [[ ! -f "$EXEC_SOURCE" ]]; then
    echo "Error: Executable file not found: $EXEC_SOURCE"
    exit 1
fi

if [[ ! -f "$ICON_SOURCE" ]]; then
    echo "Error: Icon file not found: $ICON_SOURCE"
    echo "Please ensure ico256.png exists in the icon directory"
    exit 1
fi

echo "Starting installation of $APP_NAME..."
echo "Real user: $REAL_USER"
echo "User home directory: $REAL_USER_HOME"
echo "=============================="

# 1. Create installation directory
echo "1. Creating installation directory..."
mkdir -p "$INSTALL_DIR"
if [[ $? -ne 0 ]]; then
    echo "Error: Unable to create directory $INSTALL_DIR"
    exit 1
fi

# 2. Copy executable and icon files
echo "2. Copying files..."
cp "$EXEC_SOURCE" "$INSTALL_DIR/$APP_NAME"
chmod +x "$INSTALL_DIR/$APP_NAME"
cp "$ICON_SOURCE" "$INSTALL_DIR/icon.png"

# 3. Install icon to system icon directory
echo "3. Installing icon to system directory..."
mkdir -p "$ICON_DIR"
ICON_NAME="${APP_NAME,,}"
cp "$ICON_SOURCE" "$ICON_DIR/$ICON_NAME.png"

# Create different size icon directories and use corresponding size icon files
echo "4. Creating multi-size icons..."
for size in 16 22 24 32 48 64 128 256 512; do
    size_dir="/usr/share/icons/hicolor/${size}x${size}/apps"
    mkdir -p "$size_dir"

    icon_source_file="$ICON_DIR_SOURCE/ico${size}.png"

    if [[ -f "$icon_source_file" ]]; then
        echo "  Installing ${size}x${size} icon"
        cp "$icon_source_file" "$size_dir/$ICON_NAME.png"
    elif [[ $size -eq 512 ]] && [[ -f "$ICON_SOURCE" ]]; then
        if command -v convert &> /dev/null; then
            convert "$ICON_SOURCE" -resize 512x512 "$size_dir/$ICON_NAME.png"
        else
            cp "$ICON_SOURCE" "$size_dir/$ICON_NAME.png"
        fi
    elif [[ -f "$ICON_SOURCE" ]]; then
        # Use 256x256 as base
        if command -v convert &> /dev/null; then
            convert "$ICON_SOURCE" -resize ${size}x${size} "$size_dir/$ICON_NAME.png"
        else
            cp "$ICON_SOURCE" "$size_dir/$ICON_NAME.png"
        fi
    fi

    # Set correct permissions
    if [[ -f "$size_dir/$ICON_NAME.png" ]]; then
        chmod 644 "$size_dir/$ICON_NAME.png"
    fi
done

# 5. Force update icon cache
echo "5. Updating icon cache..."
if command -v gtk-update-icon-cache &> /dev/null; then
    gtk-update-icon-cache -f -t /usr/share/icons/hicolor
    echo "  ✓ System icon cache updated"
fi

# 6. Create symbolic link to /usr/local/bin
echo "6. Creating command line link..."
ln -sf "$INSTALL_DIR/$APP_NAME" "$BIN_DIR/$APP_NAME"

# 7. Create system desktop launcher
echo "7. Creating system desktop launcher..."
DESKTOP_FILE="$DESKTOP_DIR/$ICON_NAME.desktop"

cat > "$DESKTOP_FILE" << EOF
[Desktop Entry]
Version=1.0
Type=Application
Name=$APP_NAME
GenericName=$APP_NAME Analysis Tool
Comment=Data analysis and visualization tool
Exec=$INSTALL_DIR/$APP_NAME
Icon=$ICON_NAME
Terminal=false
Categories=Science;Utility;DataVisualization;
StartupNotify=true
Keywords=data;analysis;visualization;
X-GNOME-FullName=$APP_NAME Analysis Tool
MimeType=
EOF

chmod 644 "$DESKTOP_FILE"
echo "  ✓ System desktop launcher created: $DESKTOP_FILE"

# 8. Create user desktop shortcut
echo "8. Creating user desktop shortcut..."

# Determine user desktop directory
USER_DESKTOP_DIR=""
if [[ -d "$REAL_USER_HOME/桌面" ]]; then
    USER_DESKTOP_DIR="$REAL_USER_HOME/桌面"
elif [[ -d "$REAL_USER_HOME/Desktop" ]]; then
    USER_DESKTOP_DIR="$REAL_USER_HOME/Desktop"
elif command -v xdg-user-dir &> /dev/null; then
    USER_DESKTOP_DIR=$(sudo -u $REAL_USER xdg-user-dir DESKTOP 2>/dev/null)
fi

if [[ -n "$USER_DESKTOP_DIR" ]] && [[ -d "$USER_DESKTOP_DIR" ]]; then
    echo "  Detected desktop directory: $USER_DESKTOP_DIR"

    USER_DESKTOP_FILE="$USER_DESKTOP_DIR/$APP_NAME.desktop"

    # Create file as real user
    sudo -u $REAL_USER bash -c "cat > '$USER_DESKTOP_FILE'" << EOF
[Desktop Entry]
Version=1.0
Type=Application
Name=$APP_NAME
GenericName=$APP_NAME Analysis Tool
Comment=Data analysis and visualization tool
Exec=$INSTALL_DIR/$APP_NAME
Icon=$ICON_NAME
Terminal=false
Categories=Science;Utility;DataVisualization;
StartupNotify=true
Keywords=data;analysis;visualization;
X-GNOME-FullName=$APP_NAME Analysis Tool
MimeType=
EOF

    sudo -u $REAL_USER chmod +x "$USER_DESKTOP_FILE"
    echo "  ✓ User desktop shortcut created: $USER_DESKTOP_FILE"

    # Verify file
    echo "  File permissions: $(sudo -u $REAL_USER ls -la "$USER_DESKTOP_FILE" | awk '{print $1" "$3":"$4" "$9}' 2>/dev/null || echo "Verification failed")"
else
    echo "  ⚠ Warning: Unable to find user desktop directory, skipping desktop shortcut creation"
    USER_DESKTOP_FILE=""
fi

# 9. Create user local application launcher (Important: This ensures it appears in "Show Applications")
echo "9. Creating user local application launcher..."
USER_APPS_DIR="$REAL_USER_HOME/.local/share/applications"
sudo -u $REAL_USER mkdir -p "$USER_APPS_DIR" 2>/dev/null || true

USER_APPS_FILE="$USER_APPS_DIR/$ICON_NAME.desktop"

# Create file as real user
sudo -u $REAL_USER bash -c "cat > '$USER_APPS_FILE'" << EOF
[Desktop Entry]
Version=1.0
Type=Application
Name=$APP_NAME
GenericName=$APP_NAME Analysis Tool
Comment=Data analysis and visualization tool
Exec=$INSTALL_DIR/$APP_NAME
Icon=$ICON_NAME
Terminal=false
Categories=Science;Utility;DataVisualization;
StartupNotify=true
Keywords=data;analysis;visualization;
X-GNOME-FullName=$APP_NAME Analysis Tool
MimeType=
EOF

sudo -u $REAL_USER chmod 644 "$USER_APPS_FILE"
echo "  ✓ User application launcher created: $USER_APPS_FILE"
echo "    This file ensures the application appears in 'Show Applications' menu"

# Verify file
echo "  File permissions: $(sudo -u $REAL_USER ls -la "$USER_APPS_FILE" | awk '{print $1" "$3":"$4" "$9}' 2>/dev/null || echo "Verification failed")"

# 10. Update desktop database
echo "10. Updating desktop database..."
if command -v update-desktop-database &> /dev/null; then
    # Update system database
    update-desktop-database /usr/share/applications 2>/dev/null || true
    # Update user database
    sudo -u $REAL_USER update-desktop-database "$USER_APPS_DIR" 2>/dev/null || true
    echo "  ✓ Desktop database updated"
fi

# 11. Create user local icon cache
echo "11. Creating user icon cache..."
USER_ICON_DIR="$REAL_USER_HOME/.local/share/icons/hicolor"
sudo -u $REAL_USER mkdir -p "$USER_ICON_DIR/256x256/apps" 2>/dev/null || true

if [[ -f "$ICON_SOURCE" ]]; then
    sudo -u $REAL_USER cp "$ICON_SOURCE" "$USER_ICON_DIR/256x256/apps/$ICON_NAME.png" 2>/dev/null || true
    sudo -u $REAL_USER chmod 644 "$USER_ICON_DIR/256x256/apps/$ICON_NAME.png" 2>/dev/null || true
fi

# Update user icon cache
if command -v gtk-update-icon-cache &> /dev/null && [[ -d "$USER_ICON_DIR" ]]; then
    sudo -u $REAL_USER gtk-update-icon-cache -f -t "$USER_ICON_DIR" 2>/dev/null || true
    echo "  ✓ User icon cache updated"
fi

# 12. Enable GNOME desktop icons (if using GNOME)
echo "12. Configuring desktop environment..."
if [[ "$XDG_CURRENT_DESKTOP" == *"GNOME"* ]] || [[ "$XDG_CURRENT_DESKTOP" == *"ubuntu"* ]]; then
    echo "  Detected GNOME desktop environment"

    # Enable desktop icons
    sudo -u $REAL_USER gsettings set org.gnome.desktop.background show-desktop-icons true 2>/dev/null || true

    # Check and enable desktop icons extension
    if command -v gnome-extensions &> /dev/null; then
        if gnome-extensions list | grep -q "desktop-icons"; then
            sudo -u $REAL_USER gnome-extensions enable desktop-icons@csoriano 2>/dev/null || true
        fi
    fi
fi

# 13. Force refresh desktop
echo "13. Refreshing desktop..."
if [[ "$XDG_CURRENT_DESKTOP" == *"GNOME"* ]]; then
    # Method 1: Restart GNOME Shell
    if command -v dbus-send &> /dev/null; then
        sudo -u $REAL_USER dbus-send --type=method_call --dest=org.gnome.Shell /org/gnome/Shell org.gnome.Shell.Eval string:'global.reexec_self()' 2>/dev/null || true
    fi

    # Method 2: Restart Nautilus (file manager)
    sudo -u $REAL_USER pkill nautilus 2>/dev/null || true
    sudo -u $REAL_USER pkill nautilus-desktop 2>/dev/null || true
    sleep 1
    sudo -u $REAL_USER nautilus --no-default-window 2>/dev/null &
    disown
fi

echo "=============================="
echo "Installation complete!"
echo ""

# Verify installation
echo "Verifying installation:"
echo "1. Executable file: $(ls -la "$INSTALL_DIR/$APP_NAME" | awk '{print $1" "$5" bytes"}')"
echo "2. System icon: $(ls -la "/usr/share/icons/hicolor/256x256/apps/$ICON_NAME.png" 2>/dev/null | awk '{print $1" "$5" bytes"}' || echo "Not found")"
echo "3. System desktop file: $DESKTOP_FILE"
echo "4. User application launcher: $USER_APPS_FILE"

if [[ -n "$USER_DESKTOP_FILE" ]] && [[ -f "$USER_DESKTOP_FILE" ]]; then
    echo "5. User desktop shortcut: $USER_DESKTOP_FILE"
    echo "   Status: $(sudo -u $REAL_USER ls -la "$USER_DESKTOP_FILE" | awk '{print $1" "$3":"$4}' 2>/dev/null || echo "Unknown")"
fi

echo ""
echo "The application should appear in the following locations:"
echo "- Application menu: Click 'Show Applications' at bottom left, search for '$APP_NAME'"
echo "- Command line: Type '$APP_NAME'"
if [[ -n "$USER_DESKTOP_FILE" ]]; then
    echo "- Desktop icon: Double-click '$APP_NAME'"
fi

echo ""
echo "If the application does not appear in 'Show Applications' menu, please:"
echo "1. Log out and log back in"
echo "2. Or run: sudo -u $REAL_USER update-desktop-database ~/.local/share/applications"
echo "3. Or run: nautilus -q && sleep 2 && nautilus --no-default-window &"