#!/bin/bash
# BjAnalyzer Uninstallation Script

# Check if running as root
if [[ $EUID -ne 0 ]]; then
    echo "This uninstallation script requires root privileges. Please run with sudo:"
    echo "sudo $0"
    exit 1
fi

# Define variables
APP_NAME="BjAnalyzer"
ICON_NAME="${APP_NAME,,}"
INSTALL_DIR="/opt/$APP_NAME"
BIN_DIR="/usr/local/bin"
DESKTOP_DIR="/usr/share/applications"

# Get real username
REAL_USER=${SUDO_USER:-$(logname 2>/dev/null)}
if [[ -z "$REAL_USER" ]]; then
    REAL_USER=$(who am i | awk '{print $1}')
fi
if [[ -z "$REAL_USER" ]] || [[ "$REAL_USER" == "root" ]]; then
    echo "Warning: Unable to determine real username, using current user"
    REAL_USER=$(whoami)
fi

REAL_USER_HOME=$(eval echo ~$REAL_USER)

echo "Starting uninstallation of $APP_NAME..."
echo "Real user: $REAL_USER"
echo "User home directory: $REAL_USER_HOME"
echo "=============================="

# 1. Remove system icons
echo "1. Removing system icons..."
for size in 16 22 24 32 48 64 128 256 512; do
    size_dir="/usr/share/icons/hicolor/${size}x${size}/apps"
    icon_file="$size_dir/$ICON_NAME.png"
    if [[ -f "$icon_file" ]]; then
        rm -f "$icon_file"
        echo "  ✓ Removed $size_dir/$ICON_NAME.png"
    fi
done

# 2. Remove user local icons
echo "2. Removing user local icons..."
USER_ICON_DIR="$REAL_USER_HOME/.local/share/icons/hicolor"
for size in 256; do
    user_icon_file="$USER_ICON_DIR/${size}x${size}/apps/$ICON_NAME.png"
    if [[ -f "$user_icon_file" ]]; then
        sudo -u $REAL_USER rm -f "$user_icon_file" 2>/dev/null || rm -f "$user_icon_file"
        echo "  ✓ Removed user icon $user_icon_file"
    fi
done

# 3. Update icon cache
echo "3. Updating icon cache..."
if command -v gtk-update-icon-cache &> /dev/null; then
    # Update system icon cache
    gtk-update-icon-cache -f -t /usr/share/icons/hicolor 2>/dev/null || true

    # Update user icon cache
    if [[ -d "$USER_ICON_DIR" ]]; then
        sudo -u $REAL_USER gtk-update-icon-cache -f -t "$USER_ICON_DIR" 2>/dev/null || true
    fi
    echo "  ✓ Icon cache updated"
fi

# 4. Remove symbolic link
echo "4. Removing command line link..."
if [[ -L "$BIN_DIR/$APP_NAME" ]]; then
    rm -f "$BIN_DIR/$APP_NAME"
    echo "  ✓ Removed symbolic link $BIN_DIR/$APP_NAME"
fi

# 5. Remove system desktop launcher
echo "5. Removing system desktop launcher..."
SYSTEM_DESKTOP_FILE="$DESKTOP_DIR/$ICON_NAME.desktop"
if [[ -f "$SYSTEM_DESKTOP_FILE" ]]; then
    rm -f "$SYSTEM_DESKTOP_FILE"
    echo "  ✓ Removed system desktop file $SYSTEM_DESKTOP_FILE"
fi

# 6. Remove user desktop shortcuts
echo "6. Removing user desktop shortcuts..."
# Check possible desktop directories
DESKTOP_PATHS=(
    "$REAL_USER_HOME/桌面/$APP_NAME.desktop"
    "$REAL_USER_HOME/Desktop/$APP_NAME.desktop"
)

if command -v xdg-user-dir &> /dev/null; then
    USER_DESKTOP_DIR=$(sudo -u $REAL_USER xdg-user-dir DESKTOP 2>/dev/null)
    if [[ -n "$USER_DESKTOP_DIR" ]]; then
        DESKTOP_PATHS+=("$USER_DESKTOP_DIR/$APP_NAME.desktop")
    fi
fi

for desktop_file in "${DESKTOP_PATHS[@]}"; do
    if [[ -f "$desktop_file" ]]; then
        sudo -u $REAL_USER rm -f "$desktop_file" 2>/dev/null || rm -f "$desktop_file"
        echo "  ✓ Removed desktop shortcut $desktop_file"
    fi
done

# 7. Remove user local application launcher
echo "7. Removing user local application launcher..."
USER_APPS_DIR="$REAL_USER_HOME/.local/share/applications"
USER_APPS_FILE="$USER_APPS_DIR/$ICON_NAME.desktop"

if [[ -f "$USER_APPS_FILE" ]]; then
    sudo -u $REAL_USER rm -f "$USER_APPS_FILE" 2>/dev/null || rm -f "$USER_APPS_FILE"
    echo "  ✓ Removed user application launcher $USER_APPS_FILE"
fi

# 8. Update desktop database
echo "8. Updating desktop database..."
if command -v update-desktop-database &> /dev/null; then
    # Update system database
    update-desktop-database /usr/share/applications 2>/dev/null || true
    # Update user database
    if [[ -d "$USER_APPS_DIR" ]]; then
        sudo -u $REAL_USER update-desktop-database "$USER_APPS_DIR" 2>/dev/null || true
    fi
    echo "  ✓ Desktop database updated"
fi

# 9. Remove installation directory
echo "9. Removing installation directory..."
if [[ -d "$INSTALL_DIR" ]]; then
    rm -rf "$INSTALL_DIR"
    echo "  ✓ Removed installation directory $INSTALL_DIR"
else
    echo "  ⚠ Installation directory does not exist: $INSTALL_DIR"
fi

# 10. Clean up empty directories
echo "10. Cleaning up empty directories..."
# Clean up possible empty icon directories
for size in 16 22 24 32 48 64 128 256 512; do
    size_dir="/usr/share/icons/hicolor/${size}x${size}/apps"
    if [[ -d "$size_dir" ]] && [[ -z "$(ls -A "$size_dir" 2>/dev/null)" ]]; then
        rmdir "$size_dir" 2>/dev/null || true
    fi
done

# Clean up user icon directory
if [[ -d "$USER_ICON_DIR" ]]; then
    find "$USER_ICON_DIR" -type d -empty -delete 2>/dev/null || true
fi

# 11. Refresh desktop environment
echo "11. Refreshing desktop..."
if [[ "$XDG_CURRENT_DESKTOP" == *"GNOME"* ]]; then
    # Restart Nautilus (file manager)
    sudo -u $REAL_USER pkill nautilus 2>/dev/null || true
    sudo -u $REAL_USER pkill nautilus-desktop 2>/dev/null || true
    sleep 1
    sudo -u $REAL_USER nautilus --no-default-window 2>/dev/null &
    disown
    echo "  ✓ Desktop refreshed"
fi

echo "=============================="
echo "Uninstallation complete!"
echo ""

# Verify uninstallation
echo "Verifying uninstallation results:"
echo "1. Installation directory: $(if [[ -d "$INSTALL_DIR" ]]; then echo "Still exists"; else echo "Removed"; fi)"
echo "2. Command line link: $(if [[ -L "$BIN_DIR/$APP_NAME" ]]; then echo "Still exists"; else echo "Removed"; fi)"
echo "3. System desktop file: $(if [[ -f "$SYSTEM_DESKTOP_FILE" ]]; then echo "Still exists"; else echo "Removed"; fi)"
echo "4. User application launcher: $(if [[ -f "$USER_APPS_FILE" ]]; then echo "Still exists"; else echo "Removed"; fi)"

echo ""
echo "Note:"
echo "1. You may need to log out and log back in for all changes to take effect"
echo "2. User configuration files and data (if any) have not been removed"
echo "3. For complete cleanup, please manually check the following directories:"
echo "   - $REAL_USER_HOME/.config/$APP_NAME"
echo "   - $REAL_USER_HOME/.local/share/$APP_NAME"
echo "   - $REAL_USER_HOME/.cache/$APP_NAME"